<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Hex2RGB | Minimal Popover</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600&family=JetBrains+Mono:wght@400;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --bg: #09090b;
            --card-bg: #18181b;
            --border: #27272a;
            --accent: #3b82f6;
            --text-main: #e4e4e7;
            --text-muted: #a1a1aa;
        }

        * { margin: 0; padding: 0; box-sizing: border-box; user-select: none; }

        body {
            background-color: var(--bg);
            /* Grid Background */
            background-image: linear-gradient(var(--border) 1px, transparent 1px),
            linear-gradient(90deg, var(--border) 1px, transparent 1px);
            background-size: 40px 40px;
            background-position: center center;
            font-family: 'Inter', sans-serif;
            color: var(--text-main);
            height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        /* Vignette */
        body::after {
            content: '';
            position: absolute;
            top: 0; left: 0; width: 100%; height: 100%;
            background: radial-gradient(circle at center, transparent 0%, var(--bg) 80%);
            pointer-events: none;
        }

        .app-container {
            position: relative;
            z-index: 10;
            width: 100%;
            max-width: 400px;
            background: var(--card-bg);
            border: 1px solid var(--border);
            border-radius: 16px;
            box-shadow: 0 20px 50px -10px rgba(0,0,0,0.6);
            padding: 2rem;
        }

        header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 2rem;
            border-bottom: 1px solid var(--border);
            padding-bottom: 1rem;
        }
        h1 { font-size: 1rem; font-weight: 600; letter-spacing: -0.5px; }
        .badge { font-size: 0.7rem; background: #27272a; padding: 4px 8px; border-radius: 6px; color: var(--text-muted); font-family: 'JetBrains Mono', monospace; }

        /* --- INPUT ALANI VE POPUP TETİKLEYİCİ --- */
        .input-group {
            position: relative; /* Popup buna göre hizalanacak */
            display: flex;
            align-items: center;
            background: #000;
            border: 1px solid var(--border);
            border-radius: 12px; /* Yumuşak kenarlar */
            padding: 8px 12px;
            margin-bottom: 2rem;
            transition: border-color 0.2s;
        }
        .input-group:focus-within { border-color: var(--accent); }

        .hash { font-family: 'JetBrains Mono', monospace; color: var(--text-muted); margin-right: 10px; }
        
        input[type="text"] {
            background: transparent;
            border: none;
            color: #fff;
            font-family: 'JetBrains Mono', monospace;
            font-size: 1.4rem;
            width: 100%;
            text-transform: uppercase;
            outline: none;
        }

        /* Tetikleyici Buton (Küçük Kare) */
        .color-trigger {
            width: 42px;
            height: 42px;
            border-radius: 10px; /* Kenarları yumuşattım */
            border: 2px solid #3f3f46;
            cursor: pointer;
            transition: transform 0.2s, box-shadow 0.2s;
            flex-shrink: 0;
        }
        .color-trigger:hover { transform: scale(1.05); border-color: #fff; }

        /* --- MODERN POPUP PICKER (GİZLİ) --- */
        .popover-picker {
            display: none; /* Başlangıçta gizli */
            position: absolute;
            top: 110%; /* Inputun hemen altı */
            right: 0;
            width: 240px;
            background: #18181b;
            border: 1px solid #3f3f46;
            border-radius: 24px; /* İSTEDİĞİN O YUMUŞAKLIK BURADA */
            padding: 15px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.8);
            z-index: 100;
            animation: popIn 0.2s cubic-bezier(0.16, 1, 0.3, 1);
        }

        .popover-picker.active { display: block; }

        @keyframes popIn {
            from { opacity: 0; transform: translateY(-10px) scale(0.95); }
            to { opacity: 1; transform: translateY(0) scale(1); }
        }

        /* Saturation Box */
        .sat-box {
            width: 100%;
            height: 120px;
            border-radius: 16px; /* İç köşeler de yumuşak */
            margin-bottom: 12px;
            position: relative;
            cursor: crosshair;
            background-color: red;
            background-image: linear-gradient(to right, #fff, transparent), linear-gradient(to top, #000, transparent);
            overflow: hidden;
        }

        .sat-cursor {
            width: 12px; height: 12px;
            border: 2px solid white;
            border-radius: 50%;
            position: absolute;
            transform: translate(-50%, -50%);
            pointer-events: none;
            box-shadow: 0 1px 3px rgba(0,0,0,0.5);
        }

        /* Hue Slider */
        .hue-slider {
            width: 100%;
            height: 12px;
            border-radius: 10px; /* Hap şeklinde */
            position: relative;
            cursor: pointer;
            background: linear-gradient(to right, #f00 0%, #ff0 17%, #0f0 33%, #0ff 50%, #00f 67%, #f0f 83%, #f00 100%);
        }

        .hue-cursor {
            width: 12px; height: 12px;
            background: white;
            border-radius: 50%;
            position: absolute;
            top: 50%;
            transform: translate(-50%, -50%);
            pointer-events: none;
            box-shadow: 0 1px 3px rgba(0,0,0,0.3);
        }

        /* --- SONUÇLAR --- */
        .result-rows {
            display: flex;
            flex-direction: column;
            gap: 1px;
            background: var(--border);
            border-radius: 12px;
            overflow: hidden;
            border: 1px solid var(--border);
        }

        .row {
            display: flex;
            justify-content: space-between;
            padding: 14px 20px;
            background: var(--card-bg);
            cursor: pointer;
            transition: background 0.2s;
        }
        .row:hover { background: #27272a; }

        .label { font-size: 0.8rem; color: var(--text-muted); font-weight: 600; }
        .val { font-family: 'JetBrains Mono', monospace; color: var(--accent); font-weight: 700; }

        .copy-all {
            margin-top: 1.5rem;
            width: 100%;
            background: var(--text-main);
            border: none;
            padding: 14px;
            border-radius: 12px;
            font-weight: 600;
            font-size: 0.95rem;
            cursor: pointer;
            transition: all 0.2s;
        }
        .copy-all:hover { transform: translateY(-2px); box-shadow: 0 5px 15px rgba(255,255,255,0.1); }
        .copy-all:active { transform: scale(0.98); }

    </style>
</head>
<body>

<div class="app-container">
    <header>
        <h1>HEX / RGB</h1>
        <span class="badge">V3.1 SOFT</span>
    </header>

    <div class="input-group">
        <span class="hash">#</span>
        <input type="text" id="hexInput" value="3B82F6" maxlength="6">
        
        <div class="color-trigger" id="colorTrigger" style="background: #3B82F6;"></div>

        <div class="popover-picker" id="popover">
            <div class="sat-box" id="satBox">
                <div class="sat-cursor" id="satCursor" style="top: 0; left: 100%;"></div>
            </div>
            <div class="hue-slider" id="hueSlider">
                <div class="hue-cursor" id="hueCursor" style="left: 60%;"></div>
            </div>
        </div>
    </div>

    <div class="result-rows">
        <div class="row" onclick="copyText('r')">
            <span class="label">RED</span>
            <span class="val" id="rVal">59</span>
        </div>
        <div class="row" onclick="copyText('g')">
            <span class="label">GREEN</span>
            <span class="val" id="gVal">130</span>
        </div>
        <div class="row" onclick="copyText('b')">
            <span class="label">BLUE</span>
            <span class="val" id="bVal">246</span>
        </div>
    </div>

    <button class="copy-all" id="mainCopyBtn">Kopyala</button>
</div>

<script>
    // --- Elemanlar ---
    const hexInput = document.getElementById('hexInput');
    const colorTrigger = document.getElementById('colorTrigger');
    const popover = document.getElementById('popover');
    
    // Picker Elemanları
    const satBox = document.getElementById('satBox');
    const satCursor = document.getElementById('satCursor');
    const hueSlider = document.getElementById('hueSlider');
    const hueCursor = document.getElementById('hueCursor');

    // Değer Elemanları
    const rVal = document.getElementById('rVal');
    const gVal = document.getElementById('gVal');
    const bVal = document.getElementById('bVal');
    const mainCopyBtn = document.getElementById('mainCopyBtn');

    // State
    let state = { h: 210, s: 75, v: 96, r: 59, g: 130, b: 246, hex: '3B82F6' };

    // --- Helper Logic ---
    function hsvToRgb(h, s, v) {
        s /= 100; v /= 100;
        let c = v * s, x = c * (1 - Math.abs(((h / 60) % 2) - 1)), m = v - c, r = 0, g = 0, b = 0;
        if (0 <= h && h < 60) { r = c; g = x; b = 0; }
        else if (60 <= h && h < 120) { r = x; g = c; b = 0; }
        else if (120 <= h && h < 180) { r = 0; g = c; b = x; }
        else if (180 <= h && h < 240) { r = 0; g = x; b = c; }
        else if (240 <= h && h < 300) { r = x; g = 0; b = c; }
        else if (300 <= h && h < 360) { r = c; g = 0; b = x; }
        return { r: Math.round((r + m) * 255), g: Math.round((g + m) * 255), b: Math.round((b + m) * 255) };
    }

    function rgbToHex(r, g, b) {
        return ((1 << 24) + (r << 16) + (g << 8) + b).toString(16).slice(1).toUpperCase();
    }

    function hexToRgb(hex) {
        const res = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
        return res ? { r: parseInt(res[1], 16), g: parseInt(res[2], 16), b: parseInt(res[3], 16) } : null;
    }

    // --- Core Update ---
    function updateFromHSV() {
        const rgb = hsvToRgb(state.h, state.s, state.v);
        state.r = rgb.r; state.g = rgb.g; state.b = rgb.b;
        state.hex = rgbToHex(rgb.r, rgb.g, rgb.b);
        updateUI();
    }

    function updateUI() {
        hexInput.value = state.hex;
        colorTrigger.style.background = "#" + state.hex;
        rVal.innerText = state.r; gVal.innerText = state.g; bVal.innerText = state.b;
        document.documentElement.style.setProperty('--accent', "#" + state.hex);
        
        // Saturation box rengini Hue'ya göre ayarla
        const base = hsvToRgb(state.h, 100, 100);
        satBox.style.backgroundColor = `rgb(${base.r}, ${base.g}, ${base.b})`;
    }

    // --- Interaction ---
    
    // Popup Aç/Kapa
    colorTrigger.addEventListener('click', (e) => {
        e.stopPropagation();
        popover.classList.toggle('active');
    });

    // Dışarı tıklayınca kapat
    document.addEventListener('click', (e) => {
        if (!popover.contains(e.target) && e.target !== colorTrigger) {
            popover.classList.remove('active');
        }
    });

    // Picker Logic (Drag)
    let isDragHue = false, isDragSat = false;

    function handleHue(e) {
        const rect = hueSlider.getBoundingClientRect();
        let x = e.clientX - rect.left;
        x = Math.max(0, Math.min(x, rect.width));
        hueCursor.style.left = x + 'px';
        state.h = (x / rect.width) * 360;
        updateFromHSV();
    }

    function handleSat(e) {
        const rect = satBox.getBoundingClientRect();
        let x = e.clientX - rect.left;
        let y = e.clientY - rect.top;
        x = Math.max(0, Math.min(x, rect.width));
        y = Math.max(0, Math.min(y, rect.height));
        
        satCursor.style.left = x + 'px';
        satCursor.style.top = y + 'px';
        
        state.s = (x / rect.width) * 100;
        state.v = 100 - ((y / rect.height) * 100);
        updateFromHSV();
    }

    hueSlider.addEventListener('mousedown', (e) => { isDragHue = true; handleHue(e); });
    satBox.addEventListener('mousedown', (e) => { isDragSat = true; handleSat(e); });

    window.addEventListener('mousemove', (e) => {
        if (isDragHue) handleHue(e);
        if (isDragSat) handleSat(e);
    });

    window.addEventListener('mouseup', () => { isDragHue = false; isDragSat = false; });

    // Hex Input Change
    hexInput.addEventListener('input', (e) => {
        let val = e.target.value;
        if(/^[0-9A-Fa-f]{6}$/.test(val)) {
            const rgb = hexToRgb(val);
            state.r = rgb.r; state.g = rgb.g; state.b = rgb.b; state.hex = val.toUpperCase();
            updateUI();
            // Not: HSV cursor pozisyonlarını güncellemek biraz kompleks matematik gerektirir, 
            // UX için zorunlu değil ama renk değişir.
        }
    });

    // Kopyalama
    function copyText(type) {
        let txt = (type === 'r') ? state.r : (type === 'g') ? state.g : state.b;
        navigator.clipboard.writeText(txt);
        // Basit animasyon
        const el = document.getElementById(type+'Val');
        el.style.color = '#fff';
        setTimeout(() => el.style.color = '', 200);
    }

    mainCopyBtn.addEventListener('click', () => {
        navigator.clipboard.writeText(`rgb(${state.r}, ${state.g}, ${state.b})`);
        let old = mainCopyBtn.innerText;
        mainCopyBtn.innerText = "Kopyalandı!";
        setTimeout(() => mainCopyBtn.innerText = old, 1500);
    });

</script>
</body>
</html>