<?php
/**
 * Yeni Görev Ekleme İşlemi
 * AJAX POST isteği ile çalışır
 */

header('Content-Type: application/json; charset=utf-8');

require_once '../config/db.php';

// Sadece POST isteklerini kabul et
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Sadece POST metodu kabul edilir']);
    exit;
}

// JSON verisini al
$input = json_decode(file_get_contents('php://input'), true);

// Form data veya JSON kontrolü
$title = $input['title'] ?? $_POST['title'] ?? '';
$description = $input['description'] ?? $_POST['description'] ?? '';
$status = $input['status'] ?? $_POST['status'] ?? 'todo';
$priority = $input['priority'] ?? $_POST['priority'] ?? 'medium';
$color = $input['color'] ?? $_POST['color'] ?? '#6366f1';

// Validasyon
if (empty($title)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Görev başlığı zorunludur']);
    exit;
}

// Başlık uzunluk kontrolü
if (strlen($title) > 255) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Başlık 255 karakterden uzun olamaz']);
    exit;
}

// Status validasyonu
$validStatuses = ['todo', 'inprogress', 'done'];
if (!in_array($status, $validStatuses)) {
    $status = 'todo';
}

// Priority validasyonu
$validPriorities = ['low', 'medium', 'high'];
if (!in_array($priority, $validPriorities)) {
    $priority = 'medium';
}

// Renk validasyonu (hex formatı)
if (!preg_match('/^#[a-fA-F0-9]{6}$/', $color)) {
    $color = '#6366f1';
}

try {
    $pdo = getDBConnection();
    
    // Son pozisyonu bul
    $stmt = $pdo->prepare("SELECT MAX(position) as max_pos FROM tasks WHERE status = ?");
    $stmt->execute([$status]);
    $result = $stmt->fetch();
    $position = ($result['max_pos'] ?? -1) + 1;
    
    // Görevi ekle
    $stmt = $pdo->prepare("
        INSERT INTO tasks (title, description, status, priority, color, position) 
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        sanitize($title),
        sanitize($description),
        $status,
        $priority,
        $color,
        $position
    ]);
    
    $taskId = $pdo->lastInsertId();
    
    // Eklenen görevi getir
    $stmt = $pdo->prepare("SELECT * FROM tasks WHERE id = ?");
    $stmt->execute([$taskId]);
    $task = $stmt->fetch();
    
    echo json_encode([
        'success' => true,
        'message' => 'Görev başarıyla eklendi',
        'task' => $task
    ]);
    
} catch (PDOException $e) {
    error_log("Görev ekleme hatası: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Veritabanı hatası oluştu']);
}
?>