<?php
/**
 * Görev Durumu Güncelleme İşlemi
 * Sürükle-bırak için AJAX POST/PUT isteği ile çalışır
 */

header('Content-Type: application/json; charset=utf-8');

require_once '../config/db.php';

// POST veya PUT isteklerini kabul et
if (!in_array($_SERVER['REQUEST_METHOD'], ['POST', 'PUT'])) {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Sadece POST/PUT metodu kabul edilir']);
    exit;
}

// JSON verisini al
$input = json_decode(file_get_contents('php://input'), true);

// Parametreleri al
$taskId = $input['task_id'] ?? $_POST['task_id'] ?? null;
$newStatus = $input['status'] ?? $_POST['status'] ?? null;
$newPosition = $input['position'] ?? $_POST['position'] ?? null;

// Validasyon
if (empty($taskId)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Görev ID zorunludur']);
    exit;
}

// Status validasyonu
$validStatuses = ['todo', 'inprogress', 'done'];
if ($newStatus !== null && !in_array($newStatus, $validStatuses)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Geçersiz durum değeri']);
    exit;
}

try {
    $pdo = getDBConnection();
    
    // Görevin var olduğunu kontrol et
    $stmt = $pdo->prepare("SELECT * FROM tasks WHERE id = ?");
    $stmt->execute([$taskId]);
    $task = $stmt->fetch();
    
    if (!$task) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Görev bulunamadı']);
        exit;
    }
    
    $oldStatus = $task['status'];
    $oldPosition = $task['position'];
    
    // Transaction başlat
    $pdo->beginTransaction();
    
    try {
        // Durum değişiyor mu?
        if ($newStatus !== null && $newStatus !== $oldStatus) {
            // Eski kolondan pozisyonları güncelle
            $stmt = $pdo->prepare("
                UPDATE tasks 
                SET position = position - 1 
                WHERE status = ? AND position > ?
            ");
            $stmt->execute([$oldStatus, $oldPosition]);
            
            // Yeni kolondaki pozisyonu hesapla
            if ($newPosition === null) {
                $stmt = $pdo->prepare("SELECT MAX(position) as max_pos FROM tasks WHERE status = ?");
                $stmt->execute([$newStatus]);
                $result = $stmt->fetch();
                $newPosition = ($result['max_pos'] ?? -1) + 1;
            } else {
                // Yeni kolonda pozisyonları kaydır
                $stmt = $pdo->prepare("
                    UPDATE tasks 
                    SET position = position + 1 
                    WHERE status = ? AND position >= ?
                ");
                $stmt->execute([$newStatus, $newPosition]);
            }
            
            // Görevi güncelle
            $stmt = $pdo->prepare("
                UPDATE tasks 
                SET status = ?, position = ?, updated_at = CURRENT_TIMESTAMP 
                WHERE id = ?
            ");
            $stmt->execute([$newStatus, $newPosition, $taskId]);
            
        } elseif ($newPosition !== null && $newPosition !== $oldPosition) {
            // Sadece pozisyon değişiyor (aynı kolon içinde)
            $status = $newStatus ?? $oldStatus;
            
            if ($newPosition > $oldPosition) {
                // Aşağı taşınıyor
                $stmt = $pdo->prepare("
                    UPDATE tasks 
                    SET position = position - 1 
                    WHERE status = ? AND position > ? AND position <= ?
                ");
                $stmt->execute([$status, $oldPosition, $newPosition]);
            } else {
                // Yukarı taşınıyor
                $stmt = $pdo->prepare("
                    UPDATE tasks 
                    SET position = position + 1 
                    WHERE status = ? AND position >= ? AND position < ?
                ");
                $stmt->execute([$status, $newPosition, $oldPosition]);
            }
            
            // Görevi güncelle
            $stmt = $pdo->prepare("
                UPDATE tasks 
                SET position = ?, updated_at = CURRENT_TIMESTAMP 
                WHERE id = ?
            ");
            $stmt->execute([$newPosition, $taskId]);
        }
        
        $pdo->commit();
        
        // Güncellenmiş görevi getir
        $stmt = $pdo->prepare("SELECT * FROM tasks WHERE id = ?");
        $stmt->execute([$taskId]);
        $updatedTask = $stmt->fetch();
        
        echo json_encode([
            'success' => true,
            'message' => 'Görev başarıyla güncellendi',
            'task' => $updatedTask
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
    
} catch (PDOException $e) {
    error_log("Görev güncelleme hatası: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Veritabanı hatası oluştu']);
}
?>