document.addEventListener('DOMContentLoaded', function() {
    initDragAndDrop();
    initFormSubmit();
    initTaskActions();
    updateTaskCounts();
});

// Sürükle Bırak
let draggedCard = null;

function initDragAndDrop() {
    document.querySelectorAll('.task-card').forEach(card => {
        card.addEventListener('dragstart', handleDragStart);
        card.addEventListener('dragend', handleDragEnd);
    });
    
    document.querySelectorAll('.task-list').forEach(list => {
        list.addEventListener('dragover', handleDragOver);
        list.addEventListener('drop', handleDrop);
        list.addEventListener('dragenter', () => list.closest('.column').classList.add('drag-over'));
        list.addEventListener('dragleave', (e) => {
            if (!list.contains(e.relatedTarget)) {
                list.closest('.column').classList.remove('drag-over');
            }
        });
    });
}

function handleDragStart(e) {
    draggedCard = this;
    this.classList.add('dragging');
    e.dataTransfer.effectAllowed = 'move';
}

function handleDragEnd() {
    this.classList.remove('dragging');
    document.querySelectorAll('.column').forEach(c => c.classList.remove('drag-over'));
    draggedCard = null;
}

function handleDragOver(e) {
    e.preventDefault();
    if (!draggedCard) return;
    
    const afterElement = getDragAfterElement(this, e.clientY);
    if (afterElement) {
        this.insertBefore(draggedCard, afterElement);
    } else {
        this.appendChild(draggedCard);
    }
}

function handleDrop(e) {
    e.preventDefault();
    this.closest('.column').classList.remove('drag-over');
    
    if (!draggedCard) return;
    
    const taskId = draggedCard.dataset.taskId;
    const newStatus = this.dataset.status;
    const cards = [...this.querySelectorAll('.task-card')];
    const newPosition = cards.indexOf(draggedCard);
    
    fetch('actions/update_status.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ task_id: taskId, status: newStatus, position: newPosition })
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            showToast('Görev taşındı');
            updateTaskCounts();
        }
    });
}

function getDragAfterElement(container, y) {
    const elements = [...container.querySelectorAll('.task-card:not(.dragging)')];
    
    return elements.reduce((closest, child) => {
        const box = child.getBoundingClientRect();
        const offset = y - box.top - box.height / 2;
        if (offset < 0 && offset > closest.offset) {
            return { offset, element: child };
        }
        return closest;
    }, { offset: -Infinity }).element;
}

// Form Submit
function initFormSubmit() {
    const form = document.getElementById('addTaskForm');
    if (!form) return;
    
    form.addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const btn = form.querySelector('button[type="submit"]');
        btn.disabled = true;
        btn.textContent = '...';
        
        const data = {
            title: form.title.value.trim(),
            description: form.description.value.trim(),
            priority: form.priority.value,
            color: form.color.value
        };
        
        if (!data.title) {
            showToast('Görev başlığı gerekli', true);
            btn.disabled = false;
            btn.textContent = 'Ekle';
            return;
        }
        
        try {
            const res = await fetch('actions/add_task.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(data)
            });
            const result = await res.json();
            
            if (result.success) {
                showToast('Görev eklendi');
                form.reset();
                form.color.value = '#3b82f6';
                addTaskToBoard(result.task);
                updateTaskCounts();
            } else {
                showToast(result.message || 'Hata oluştu', true);
            }
        } catch (err) {
            showToast('Bağlantı hatası', true);
        }
        
        btn.disabled = false;
        btn.textContent = 'Ekle';
    });
}

function addTaskToBoard(task) {
    const list = document.querySelector(`.task-list[data-status="${task.status}"]`);
    if (!list) return;
    
    const empty = list.querySelector('.empty');
    if (empty) empty.remove();
    
    const priorityLabels = { low: 'Düşük', medium: 'Orta', high: 'Yüksek' };
    const date = new Date(task.created_at);
    const dateStr = date.toLocaleDateString('tr-TR', { day: 'numeric', month: 'short' });
    
    const card = document.createElement('div');
    card.className = 'task-card';
    card.draggable = true;
    card.dataset.taskId = task.id;
    card.style.setProperty('--task-color', task.color);
    
    card.innerHTML = `
        <div class="task-top">
            <span class="task-title">${escapeHtml(task.title)}</span>
            <span class="task-priority priority-${task.priority}">${priorityLabels[task.priority]}</span>
        </div>
        ${task.description ? `<p class="task-desc">${escapeHtml(task.description)}</p>` : ''}
        <div class="task-footer">
            <span class="task-date">${dateStr}</span>
            <div class="task-actions">
                <button class="task-edit" title="Düzenle"><svg width="14" height="14" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"><path d="M11 4H4a2 2 0 00-2 2v14a2 2 0 002 2h14a2 2 0 002-2v-7"/><path d="M18.5 2.5a2.121 2.121 0 013 3L12 15l-4 1 1-4 9.5-9.5z"/></svg></button>
                <button class="task-delete" title="Sil"><svg width="14" height="14" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24"><path d="M3 6h18M8 6V4a2 2 0 012-2h4a2 2 0 012 2v2m3 0v14a2 2 0 01-2 2H7a2 2 0 01-2-2V6h14z"/></svg></button>
            </div>
        </div>
    `;
    
    card.addEventListener('dragstart', handleDragStart);
    card.addEventListener('dragend', handleDragEnd);
    
    list.appendChild(card);
}

// Task Actions
function initTaskActions() {
    document.addEventListener('click', (e) => {
        const deleteBtn = e.target.closest('.task-delete');
        const editBtn = e.target.closest('.task-edit');
        
        if (deleteBtn) {
            const card = deleteBtn.closest('.task-card');
            if (card && confirm('Bu görevi silmek istediğinize emin misiniz?')) {
                deleteTask(card);
            }
        }
        
        if (editBtn) {
            const card = editBtn.closest('.task-card');
            if (card) openEditModal(card);
        }
    });
}

async function deleteTask(card) {
    const taskId = card.dataset.taskId;
    
    try {
        const res = await fetch('actions/delete_task.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ task_id: taskId })
        });
        const data = await res.json();
        
        if (data.success) {
            card.style.opacity = '0';
            card.style.transform = 'scale(0.9)';
            card.style.transition = 'all 0.2s';
            setTimeout(() => {
                card.remove();
                updateTaskCounts();
                checkEmptyState();
            }, 200);
            showToast('Görev silindi');
        } else {
            showToast(data.message || 'Silinemedi', true);
        }
    } catch (err) {
        showToast('Bağlantı hatası', true);
    }
}

function openEditModal(card) {
    const taskId = card.dataset.taskId;
    const title = card.querySelector('.task-title').textContent;
    const descEl = card.querySelector('.task-desc');
    const desc = descEl ? descEl.textContent : '';
    const priorityClass = card.querySelector('.task-priority').className;
    const priority = priorityClass.includes('low') ? 'low' : priorityClass.includes('high') ? 'high' : 'medium';
    const color = getComputedStyle(card).getPropertyValue('--task-color').trim() || '#3b82f6';
    
    // Modal oluştur
    const modal = document.createElement('div');
    modal.className = 'modal-overlay';
    modal.innerHTML = `
        <div class="modal">
            <div class="modal-header">
                <h3>Görevi Düzenle</h3>
                <button class="modal-close">&times;</button>
            </div>
            <form class="modal-form">
                <input type="hidden" name="task_id" value="${taskId}">
                <div class="modal-field">
                    <label>Başlık</label>
                    <input type="text" name="title" value="${escapeHtml(title)}" required>
                </div>
                <div class="modal-field">
                    <label>Açıklama</label>
                    <input type="text" name="description" value="${escapeHtml(desc)}">
                </div>
                <div class="modal-row">
                    <div class="modal-field">
                        <label>Öncelik</label>
                        <select name="priority">
                            <option value="low" ${priority === 'low' ? 'selected' : ''}>Düşük</option>
                            <option value="medium" ${priority === 'medium' ? 'selected' : ''}>Orta</option>
                            <option value="high" ${priority === 'high' ? 'selected' : ''}>Yüksek</option>
                        </select>
                    </div>
                    <div class="modal-field">
                        <label>Renk</label>
                        <input type="color" name="color" value="${color}" class="color-input">
                    </div>
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn-cancel">İptal</button>
                    <button type="submit" class="btn">Kaydet</button>
                </div>
            </form>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Events
    modal.querySelector('.modal-close').onclick = () => modal.remove();
    modal.querySelector('.btn-cancel').onclick = () => modal.remove();
    modal.onclick = (e) => { if (e.target === modal) modal.remove(); };
    
    modal.querySelector('.modal-form').onsubmit = async (e) => {
        e.preventDefault();
        const form = e.target;
        
        const data = {
            task_id: form.task_id.value,
            title: form.title.value.trim(),
            description: form.description.value.trim(),
            priority: form.priority.value,
            color: form.color.value
        };
        
        try {
            const res = await fetch('actions/edit_task.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(data)
            });
            const result = await res.json();
            
            if (result.success) {
                updateCardUI(card, result.task);
                modal.remove();
                showToast('Görev güncellendi');
            } else {
                showToast(result.message || 'Güncellenemedi', true);
            }
        } catch (err) {
            showToast('Bağlantı hatası', true);
        }
    };
}

function updateCardUI(card, task) {
    const priorityLabels = { low: 'Düşük', medium: 'Orta', high: 'Yüksek' };
    
    card.querySelector('.task-title').textContent = task.title;
    card.style.setProperty('--task-color', task.color);
    
    const priorityEl = card.querySelector('.task-priority');
    priorityEl.className = `task-priority priority-${task.priority}`;
    priorityEl.textContent = priorityLabels[task.priority];
    
    let descEl = card.querySelector('.task-desc');
    if (task.description) {
        if (descEl) {
            descEl.textContent = task.description;
        } else {
            const newDesc = document.createElement('p');
            newDesc.className = 'task-desc';
            newDesc.textContent = task.description;
            card.querySelector('.task-top').after(newDesc);
        }
    } else if (descEl) {
        descEl.remove();
    }
}

// Helpers
function updateTaskCounts() {
    document.querySelectorAll('.column').forEach(col => {
        const count = col.querySelectorAll('.task-card').length;
        col.querySelector('.column-count').textContent = count;
    });
}

function checkEmptyState() {
    document.querySelectorAll('.task-list').forEach(list => {
        if (list.querySelectorAll('.task-card').length === 0 && !list.querySelector('.empty')) {
            const empty = document.createElement('div');
            empty.className = 'empty';
            empty.textContent = 'Görev yok';
            list.appendChild(empty);
        }
    });
}

function showToast(msg, isError = false) {
    const existing = document.querySelector('.toast');
    if (existing) existing.remove();
    
    const toast = document.createElement('div');
    toast.className = 'toast' + (isError ? ' error' : '');
    toast.textContent = msg;
    document.body.appendChild(toast);
    
    setTimeout(() => toast.remove(), 2500);
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}